<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Route;
use App\Models\Log; // Adicione a importação do modelo Log

class JogoModuloController extends Controller
{
    /**
     * Recebe o POST de conclusão do jogo de qualquer módulo.
     *
     * Aceita:
     * - concluido: bool (true/1 para marcado como concluído)
     * - acertos:   int  (qtd. de pares/itens corretos)
     * - total:     int  (qtd. total de pares/itens do jogo)
     * - tempo:     int  (segundos; opcional, apenas para feedback)
     */
    public function store(Request $request, int $modulo)
    {
        $user  = Auth::user();
        $campo = 'usu_resp_mod' . (int) $modulo;

        // Campo precisa existir na tabela users
        if (!Schema::hasColumn('users', $campo)) {
            return $this->redirectToModulo($modulo)->with('error', 'Módulo inválido.');
        }

        // Validação leve (tudo opcional, mas consistente se vier)
        $validated = $request->validate([
            'concluido' => 'nullable|boolean',
            'acertos'   => 'nullable|integer|min:0|max:50',
            'total'     => 'nullable|integer|min:1|max:50',
            'tempo'     => 'nullable|integer|min:0|max:86400',
        ]);

        $concluido = (bool) ($validated['concluido'] ?? false);
        $acertos   = (int)  ($validated['acertos']   ?? 0);
        $total     = (int)  ($validated['total']     ?? 0);
        $tempo     = (int)  ($validated['tempo']     ?? 0);

        // Regra: conclui se veio "concluido=true" OU se acertos >= total (quando informado)
        $ok = $concluido || ($total > 0 && $acertos >= $total);
        $user->$campo = $ok ? 1 : 2;   // <- aqui grava usu_resp_modX = 1 quando concluir
        $user->save();

        // Mensagens
        $tempoFmt = $this->formatarTempo($tempo);
        $msgOk  = "Módulo concluído!";
        $msgErr = $total > 0
            ? "Você acertou {$acertos} de {$total}. É preciso acertar todos para concluir."
            : 'Jogo não concluído. Tente novamente.';

        // Log: Registra a ação de conclusão do módulo
        if ($ok) {
            Log::create([
                'usu_id' => $user->id,
                'log_ip' => request()->ip(),
                'log_acao' => "concluiu o módulo {$modulo}",
                'usu_email' => $user->email,
            ]);
        }

        // JSON (se for Ajax)
        if ($request->expectsJson()) {
            return response()->json([
                'success' => $ok,
                'message' => $ok ? $msgOk : $msgErr,
            ]);
        }

        // Redireciona para a página do módulo
        return $this->redirectToModulo($modulo)
            ->with($ok ? 'success' : 'error', $ok ? $msgOk : $msgErr);
    }

    /** Formata segundos como "Xs", "Xm Ys" ou "Xh Ym Zs" */
    private function formatarTempo(int $segundos): string
    {
        if ($segundos <= 0) return '0s';
        $h = intdiv($segundos, 3600);
        $m = intdiv($segundos % 3600, 60);
        $s = $segundos % 60;

        if ($h > 0) return sprintf('%dh %dm %ds', $h, $m, $s);
        if ($m > 0) return sprintf('%dm %ds', $m, $s);
        return sprintf('%ds', $s);
    }

    // Redireciona para a página do módulo
    protected function redirectToModulo(int $modulo)
    {
        if (Route::has('modulos.show')) {
            return redirect()->route('modulos.show', ['modulo' => $modulo]);
        }
        if (Route::has('modulo.show')) {
            return redirect()->route('modulo.show', ['modulo' => $modulo]);
        }
        // Seu path atual é /modulo{N}
        return redirect("/modulo{$modulo}");
    }
}
